SUBROUTINE ajuste_chi2(funcoes, y, parametros, n, m)
    !Determina os parâmetros do ajuste das funcoes fornecidas para os pontos dados pelo método dos mínimos quadrados.

    IMPLICIT NONE
    
    DOUBLE PRECISION, DIMENSION(n,m) :: funcoes !Matriz com todas as n funcoes fi calculadas para os m pontos (x,y)
    DOUBLE PRECISION, DIMENSION(n) :: parametros, vetorB !Conjunto de n parâmetros fornecido e vetorB de parâmetros livres do sistema a ser resolvido
    DOUBLE PRECISION, DIMENSION(m) :: y !Coordenada y dos pontos dados
    DOUBLE PRECISION, DIMENSION(n,n) :: matrizA !Matriz de coeficientes do sistema
    INTEGER :: n, m, i, j !Quantidades n e m de funções/parâmetros e de pontos, respectivamente, e variáveis auxiliares
    LOGICAL :: condicionamento !Variável que verifica o condicionamento do sistema formado
      
    !Construindo matrizA e vetorB do sistema (matrizA * parametros = vetorB)
    DO i = 1, n
    
        matrizA(i,i) = SUM(funcoes(i,1:n) * funcoes(i,1:n))
        
        DO j = 1, n
        
            matrizA(i,j) = SUM(funcoes(i,1:n) * funcoes(j,1:n))
            matrizA(j,i) = matrizA(i,j)
            
        END DO
        
        vetorB(i) = SUM(funcoes(i,1:n) * y)
        
    END DO
    
    !Resolve o sistema pelo método de Gauss, verificando o condicionamento do sistema
    CALL gauss(matrizA, parametros, vetorB, n, condicionamento)

END SUBROUTINE ajuste_chi2

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

DOUBLE PRECISION FUNCTION norma_vetor(array, n)
    !Calcula a norma de um dado vetor, com dada dimensao n

    IMPLICIT NONE

    INTEGER :: n
    DOUBLE PRECISION, DIMENSION(n), INTENT(IN) :: array

    !Cálculo da norma
    norma_vetor = SQRT(SUM(array * array))

END FUNCTION norma_vetor

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

SUBROUTINE eliminacao(matriz, n, array_b)
    !Recebe uma matriz de coeficientes nxn, a triangulariza por pivotamento parcial e realiza as mesmas operacoes no vetor array_b de termos independentes.
    
    IMPLICIT NONE
    
    DOUBLE PRECISION, DIMENSION(n,n) :: matriz !Matriz a ser triangularizada
    DOUBLE PRECISION, DIMENSION(n) :: aux_matriz, array_b !Array auxiliar nos cálculos e array de termos independentes
    DOUBLE PRECISION :: x, aux_array !Variáveis auxiliares de cálculos
    INTEGER :: n, i, j !Dimensão n da matriz e dos arrays acima; variáveis auxiliares dos cálculos
    
    !Laço que varre linhas da matriz
    DO i = 1, n

        !Laço que varre as linhas adjacentes
        j = i + 1    

        DO WHILE (j <= n)
        
            !Compara "candidatos" a pivô parcial
            IF (ABS(matriz(i,i)) < ABS(matriz(j,i))) THEN
            
                !Troca as linhas colocando na posição correta a qual contém o maior pivô e troca as mesmas linhas no array_b
                aux_matriz(i:n) = matriz(i,i:n)
                matriz(i,i:n) = matriz(j,i:n)
                matriz(j,i:n) = aux_matriz(i:n)
                
                aux_array = array_b(i)
                array_b(i) = array_b(j)
                array_b(j) = aux_array

            END IF

            j = j + 1

        END DO

        !Laço que faz a eliminação
        j = i + 1

        DO WHILE (j <= n)
        
            !Define o multiplicador x do pivô, faz a eliminação e aplica o multiplicador no array_b
            x = matriz(j,i) / matriz(i,i)
            matriz(j,i:n) = matriz(i,i:n) * x - matriz(j,i:n)
            array_b(j) = array_b(i) * x - array_b(j)
            
            j = j + 1

        END DO
    
    END DO

END SUBROUTINE eliminacao

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

SUBROUTINE substituicao(matriz, array_x, array_b, n)
    !Recebe uma matriz nxn e dois arrays de dimensão n. A partir da matriz e array_b, encontra a solução array_x de um sistema linear pelo método de Gauss (já feita a etapa de eliminação)

    IMPLICIT NONE

    DOUBLE PRECISION, DIMENSION(n,n) :: matriz !Matriz já triangularizada
    DOUBLE PRECISION, DIMENSION(n), INTENT(OUT) :: array_x !Array solução do sistema
    DOUBLE PRECISION, DIMENSION(n) :: array_b !Array b de termos independentes, modificado pela triangularizacao de A
    DOUBLE PRECISION :: soma, aux !Variáveis auxiliares dos cálculos
    INTEGER :: n, i, j !Dimensão n dos arrays/matriz acima e variáveis auxiliares

    !Determina os elementos do array_x por back substitution
    array_x(n) = array_b(n) / matriz(n,n)
    
    DO i = n - 1, 1, -1

        !Laço que determina a somatoria no cálculo dos elementos array_x(i)
        j = i + 1
        soma = 0.

        DO WHILE (j <= n)

            soma = soma + matriz(i,j) * array_x(j)

            j = j + 1

        END DO

        !Cálculo propriamente dito
        array_x(i) = (array_b(i) - soma) / matriz(i,i)

    END DO

END SUBROUTINE substituicao

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

SUBROUTINE gauss(matrizA, vetorX, vetorB, n, condicionamento)
    !Resolvendo um sistema linear pelo método de eliminação de Gauss

    IMPLICIT NONE

    DOUBLE PRECISION, DIMENSION(n,n) :: matrizA, matrizA_norm !Matriz de coeficientes original e normalizada
    DOUBLE PRECISION, DIMENSION(n) :: vetorX, vetorB !Vetores de incógnitas (solução) e de termos independentes
    INTEGER :: i, j, n !Variáveis auxiliares e dimensao n dos arrays/matrizes
    DOUBLE PRECISION, EXTERNAL :: norma_vetor !Função que calcula a norma de um vetor
    DOUBLE PRECISION :: mod_det !Módulo do determinante da matriz normalizada
    LOGICAL :: condicionamento !Indica o condicionamento do sistema

    !CONDICIONAMENTO DO SISTEMA
    !Primeiro: normalização da matriz de coeficientes
    DO i = 1, n
        
        matrizA_norm(i,:) = matrizA(i,:) / norma_vetor(matrizA(i,:), n)
        vetorB(i) = vetorB(i) / norma_vetor(matrizA(i,:), n)

    END DO

    !Segundo: encontrando o determinante da matriz normalizada
    CALL eliminacao(matrizA_norm, n, vetorB)
    mod_det = 1.

    DO i = 1, n

        mod_det = mod_det * matrizA_norm(i,i)

    END DO

    mod_det = ABS(mod_det)

    !Verificando condicionamento e, caso bem condicionado, resolve o sistema
    condicionamento = .FALSE.

    IF (mod_det > 1.E-3) THEN
        condicionamento = .TRUE.

        CALL substituicao(matrizA_norm, vetorX, vetorB, n)

        PRINT *, "A solução encontrada para os valores dos parâmetros a, b e c, respectivamente, foi:"
        PRINT *, vetorX

    ELSE IF (mod_det == 0.) THEN

        PRINT *, "O sistema é indeterminado."

    ELSE

        PRINT *, "O sistema é mal condicionado."

    END IF
    
END SUBROUTINE gauss

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

PROGRAM mmq
    !Calcula os valores dos parâmetros da parábola y = a + bx + cx^2 pelo método dos mínimos quadrados

    IMPLICIT NONE
    
    DOUBLE PRECISION, DIMENSION(3,11) :: funcoes !Matriz com os 3 arrays de funções fi
    DOUBLE PRECISION, DIMENSION(3) :: parametros !Conjunto de parâmetros
    DOUBLE PRECISION, DIMENSION(11) :: x, y, f1, f2, f3 !Coordenadas x e y dos pontos dados e funções fi calculadas para cada um
    INTEGER :: i !Auxiliar

    !Definindo os pontos
    x = (/0., 1., 2., 3., 4., 5., 6., 7., 8., 9., 10./)
    y = (/2.6, 0.35, -7.2, -21., -30., -60., -91., -127., -165., -213., -264./)
    
    !Definindo os arrays f1, f2, f3 e a matriz funcoes
    f1 = 1.
    f2 = x
    f3 = x * x
    
    funcoes(1, 1:11) = f1
    funcoes(2, 1:11) = f2
    funcoes(3, 1:11) = f3
    
    DO i = 1, 11
    
        PRINT *, x(i)
    
    end do
    
    print *,
    
        DO i = 1, 11
    
        PRINT *, y(i)
    
    end do
    
    print *,
    
    !Calculando o valor dos parâmetros
    CALL ajuste_chi2(funcoes, y, parametros, 3, 11)

END PROGRAM mmq
